#include "volume.h"
#include <locale.h>


int volume::RESOLUTION_SURFACES_IMPLICITES = RESOLUTION_16;


volume::volume ()
{
	startImage = espace = largeur = hauteur = nbr = -1;
	seuilMin = seuilMax = -1;
	nbrBytes = CODAGE_UNDEFINED;
	donnees = NULL;
	imageTmp = NULL;
	valeurMin = NULL;
	valeurMax = NULL;
	isBinaire = false;
	isCharge = false;
}


volume::volume (int nbrImage, int codage, int debutPremiereImage, int espaceEntreImage, int largeurImage, int hauteurImage)
{
	setInfos (nbrImage, codage, debutPremiereImage, espaceEntreImage, largeurImage, hauteurImage);
}


volume::~volume()
{
	if (isCharge)
	{
		if (donnees != NULL)
		{
			for (int i=0; i<nbr; i++)
				if (donnees[i] != NULL)
					delete donnees[i];
			
			delete [] donnees;
		}
		
		if (valeurMax)
			delete [] valeurMax;
		
		if (valeurMin)
			delete [] valeurMin;
		
		if (imageTmp)
			delete [] imageTmp;
	}
}


void volume::setInfos (int nbrImage, int codage, int debutPremiereImage, int espaceEntreImage, int largeurImage, int hauteurImage)
{
	isCharge = true;
	nbrBytes = codage;
	startImage = debutPremiereImage;
	espace = espaceEntreImage;
	largeur = largeurImage;
	hauteur = hauteurImage;
	taille = largeur * hauteur * nbrBytes;
	nbr = nbrImage; 
	try 
	{
		donnees = new unsigned char* [nbr];
		valeurMin = new unsigned int [nbr];
		valeurMax = new unsigned int [nbr];
	} 
	catch(bad_alloc ex)
	{
		cerr << "[-] volume::setInfos > " << ex.what() << '\n';
		exit(0); 
	}
	
	for (int i=0; i!=nbr; i++)
	{
		valeurMin[i] = 99999;
		valeurMax[i] = 0;
		
		try 
		{
			donnees[i] = new unsigned char [taille];
		} 
		catch(bad_alloc ex)
		{
			cerr << "[-] volume::setInfos > " << ex.what() << '\n';
			exit(0); 
		}
		
		for (int j=0; j!=taille; j++)
			donnees[i][j] = (unsigned char)0;
	}
}


int volume::getLargeur() { return largeur; }
int volume::getHauteur() { return hauteur; }
int volume::getNbr()     { return nbr;     }


void volume::chargerRAW (const char *fichierImg, int *pos)
{	
	if (nbrBytes == CODAGE_UNDEFINED)
	{
		cout << "[-] Impossible de charges les images, type de donnes inconnu.\n";
		return;
	}
	
	ifstream fichierRAW (fichierImg, ios::in | ios::binary);
	
	if (!fichierRAW.is_open())
		cout << "[-] volume::charger > Impossible d'ouvrir le fichier " << fichierImg << endl;
	else
	{
		/* on se positionne au dbut de la premiere image */
		fichierRAW.seekg(startImage, ios::beg);
		
		/* pour toutes les images */
		for (int i=0; i!=nbr; i++)
		{
			/* on lit  partir du fichier */
			for (int j=0; j<hauteur; j++)
			{
				for (int k=0; k<largeur; k++)
				{
					int val = 0;
					
					/* valeur du pixel en (j,k) */
					for (int l=0; l<nbrBytes; l++)
					{
						if (!fichierRAW.eof())
							val += val*255 + (int)fichierRAW.get();
						else
						{
							cout << "[-] volume::charger > Fichier imcomplet.\n";
							exit(-1);
						}
					}
					
					setValeurPixel(i,k,hauteur-j-1,val);
					
					if (val < (int)valeurMin[i])
						valeurMin[i] = val;
					
					if (val > (int)valeurMax[i])
						valeurMax[i] = val;
				}
			}
			
			/* incrmentation du compteur pour l'affichage */			
			if (pos != NULL)
				(*pos)++;
			
			/* on se dcale pour se placer au dbut de l'image suivante */
			fichierRAW.seekg(espace, ios::cur);
		}
		fichierRAW.close();
	}
}


void volume::charger (const char *fichierImg, int format, int *pos)
{
	isCharge = true;
	
	if (format == FORMAT_RAW)
		chargerRAW(fichierImg, pos);
	else if (format == FORMAT_DV)
		chargerDV(fichierImg, pos);
	else if (format == FORMAT_3D)
		charger3D(fichierImg, pos);
	else if (format == FORMAT_OBJ)
		chargerOBJ(fichierImg, pos);
}


void volume::sauvegarderRAW (string fichierImg, int *pos)
{
	ofstream fichierRAW (fichierImg.c_str(), ios::out);
	
	if (!fichierRAW.is_open())
		cout << "[-] volume::sauvegarderRAW > Impossible d'ouvrir le fichier " << fichierImg << endl;
	else
	{
		for (int i=0; i!=startImage; i++)
			fichierRAW << " ";
		
		/* pour toutes les images */
		for (int i=0; i!=nbr; i++)
		{
			/* on lit  partir du fichier */
			for (int j=0; j<hauteur; j++)
			{
				for (int k=0; k<largeur; k++)
				{
					for (int l=0; l<nbrBytes; l++)
						fichierRAW << (unsigned char)getValeurOctet(i,k,hauteur-j-1,l);
				}
			}
			/* incrmentation du compteur pour l'affichage */			
			if (pos != NULL)
				(*pos)++;
			
			/* on se dcale pour se placer au dbut de l'image suivante */
			for (int i=0; i!=espace; i++)
				fichierRAW << " ";
		}
		
		fichierRAW.close();
	}
}


void volume::sauvegarder (string fichierImg, int format, int *pos)
{
	if (format == FORMAT_RAW)
		sauvegarderRAW(fichierImg, pos);
	else if (format == FORMAT_DV)
		sauvegarderDV(fichierImg, pos);
	else if (format == FORMAT_3D)
		sauvegarder3D(fichierImg, pos);
	else if (format == FORMAT_OBJ)
		sauvegarderOBJ(fichierImg, pos);
}


void volume::expansionDynamique (int num, unsigned char **dst)
{
	/* classe minimum et maximum */
	int valMin, valMax;
	valMin = 9999;
	valMax = 0;
	/* variables permettant de calculer l'expansion */
	double alpha, beta;
	
	valMin = getValeurMinimale(num);
	valMax = getValeurMaximale(num);
	
	/* gestion d'erreur : impossible de diviser par zro */
	if (valMin == valMax)
	{
		alpha = 255.0 / (double)(valMax);
		beta  = - valMin * alpha;
	}
	else
	{
		alpha = 255.0 / (double)(valMax-valMin);
		beta  = - valMin * alpha;
	}
	
	if (!(*dst))
	{
		/* allocation de l'image de destination */
		(*dst) = new unsigned char[largeur*hauteur];
	}
	
	/* on parcourt toute l'image */
	for (int j=0; j<hauteur; j++)
	{
		for (int k=0; k<largeur; k++)
		{
			int val = getValeurPixel(num,k,j);
			
			if (seuilMin!=-1 && val < seuilMin)
				(*dst)[j*hauteur+k] = (unsigned char)NOIR;
			else if (seuilMax!=-1 && val > seuilMax)
				(*dst)[j*hauteur+k] = (unsigned char)NOIR;
			else
				(*dst)[j*hauteur+k] = (unsigned char)(alpha*val + beta);
		}
	}
}

	
void volume::seuillerTemporairement (int valMin, int valMax)
{
	seuilMin = valMin;
	seuilMax = valMax;
}


void volume::chercherBorne (int hauteurCoupe)
{
	valeurMin[hauteurCoupe] = 999999;
	valeurMax[hauteurCoupe] = 0;
	for (int j=0; j<largeur; j++)
	{
		for (int k=0; k<hauteur; k++)
		{
			unsigned int val = getValeurPixel(hauteurCoupe,j,k);
			
			if (val < valeurMin[hauteurCoupe])
				valeurMin[hauteurCoupe] = val;
			
			if (val > valeurMax[hauteurCoupe])
				valeurMax[hauteurCoupe] = val;
		}
	}
}
	

/* -1 = infini */
void volume::seuillerHauteurCoupe (int hauteurCoupe, int valMin, int valMax, bool binariser, int *pos)
{
	/* on parcourt toute l'image */
	for (int j=0; j<largeur; j++)
	{
		for (int k=0; k<hauteur; k++)
		{
			int val = getValeurPixel(hauteurCoupe,j,k);
			
			/* tous les pixels non compris dans les bornes sont positionns  0 */
			if ((valMin!=-1 && val < valMin) || (valMax!=-1 && val > valMax))
			{
				setValeurPixel(hauteurCoupe,j,k,NOIR);
				valeurMin[hauteurCoupe] = NOIR;
			}
			/* si on seuille on ne touche pas les autres pixels */
			else if (binariser)
			{
				if (hauteurCoupe<12)
					setValeurPixel(hauteurCoupe,j,k,NOIR);
				else
					setValeurPixel(hauteurCoupe,j,k,BLANC);
				
				valeurMax[hauteurCoupe] = BLANC;
			}
		}
	}
	
	/* si on seuille, on recherche les valeurs valMin et valMax */
	if (!binariser)
		chercherBorne(hauteurCoupe);
	
	if (pos)
		(*pos)++;
}


void volume::seuiller (int valMin, int valMax)
{
	seuilMin = seuilMax = -1;
	
	for (int i=0; i<nbr; i++)
		seuillerHauteurCoupe(i, valMin, valMax, false);
}


void volume::convertir8bits (int *pos)
{
	if (nbrBytes == 1)
		return;
	
	unsigned char **save;
	save = new unsigned char *[nbr];
	
	for (int img=0; img!=nbr; img++)
	{
		save[img] = new unsigned char[largeur*hauteur];
		if (save[img] == NULL)
		{
			cout << "[-] volume::convertir8bit > Erreur lors de l'allocation memoire.\n";
			exit(-1);
		}
		
		expansionDynamique(img, &save[img]);
		
		if (pos)
			(*pos)++;
	}
	
	
	for (int img=0; img!=nbr; img++)
		delete [] donnees[img];
	delete [] donnees;
	
	donnees = save;
	nbrBytes = 1;
	
	for (int i=0; i!=nbr; i++)
		chercherBorne(i);
}


void volume::convertirOBJ (int *pos)
{
	chercherSurfaceImplicite(".obj", pos);
}


void volume::binariser (int h, int valMin, int valMax, int *pos)
{
	seuilMin = seuilMax = -1;
	
	if (h == -1)
	{
		for (h=0; h!=nbr; h++)
			seuillerHauteurCoupe(h, valMin, valMax, true, pos);
		
		isBinaire = true;
	}
	else
		seuillerHauteurCoupe(h, valMin, valMax, true, pos);
}


void volume::appliquerFonction (double (&fonction)(double), int hauteurCoupe, int *pos)
{
	if (hauteurCoupe == -1)
	{
		for (int i=0; i!=nbr; i++)
			appliquerFonction(fonction, i, pos);
	}
	else
	{	
		double valMax = 0;
		for (int j=0; j<largeur; j++)
		{
			for (int k=0; k<hauteur; k++)
			{
				double val = getValeurPixel(hauteurCoupe,j,k);
				val /= valeurMax[hauteurCoupe]; /* valeur entre 0 et 1        */
				val  = fonction(val);           /* application de la fonction */
				val *= valeurMax[hauteurCoupe]; /* valeur entre 0 et valMax      */
				if (val > valMax)
					valMax = val;
				
				setValeurPixel(hauteurCoupe,j,k,(int)val);				
			}
		}
		chercherBorne(hauteurCoupe);
		
		if (pos)
			(*pos)++;
	}
}


void volume::inversion (int h, int *pos)
{
	appliquerFonction(fonction::inversion, h, pos);
}


void volume::carre (int h, int *pos)
{
	appliquerFonction(fonction::carre, h, pos);
}


void volume::exponentielle (int h, int *pos)
{
	appliquerFonction(fonction::exponentielle, h, pos);
}


void volume::logarithmique (int h, int *pos)
{
	appliquerFonction(fonction::logarithmique, h, pos);
}


unsigned int volume::getValeurMinimale (int hauteurCoupe)
{
	if (hauteur < 0 || hauteurCoupe >= nbr)
		return 0;
	
	return valeurMin[hauteurCoupe];
}


unsigned int volume::getValeurMaximale (int hauteurCoupe)
{
	if (hauteur < 0 || hauteurCoupe >= nbr)
		return 0;
	
	return valeurMax[hauteurCoupe];
}


void volume::filtreGaussien (int hauteurCoupe, int *pos)
{
	float sigma = 1.0;
	double ***gaussienne;
	unsigned char **ancienneImage;
	int largeurFiltre, hauteurFiltre, profondeurFiltre;
	int tailleX, tailleY, tailleZ;
	
	tailleX = tailleY = tailleZ = 3;
	largeurFiltre    = (tailleX-1)/2;
	hauteurFiltre    = (tailleY-1)/2;
	profondeurFiltre = (tailleZ-1)/2;
		
	gaussienne = new double **[tailleX];
	for (int i=0; i!=tailleX; i++)
	{
		gaussienne[i] = new double *[tailleY];
		for (int j=0; j!=tailleY; j++)
			gaussienne[i][j] = new double [tailleZ];
	}
	

	/************************ GRADIENT GAUSSIEN ************************/
	double coeffFixe, diviseur, expo, sommeFiltre;
	diviseur = -2*(sigma*sigma);
	sommeFiltre = 0;
	
	for (int i=0; i!=tailleX; i++)
	{
		for (int j=0; j!=tailleY; j++)
		{
			for (int k=0; k!=tailleZ; k++)
			{
				coeffFixe  = 1 / ( pow(sigma,3)*2*PI*sqrt(2*PI) );
				
				expo = pow(i-largeurFiltre, 2) + pow(j-hauteurFiltre, 2) + pow(k-profondeurFiltre,2);
				expo /= diviseur;
				gaussienne[i][j][k] = coeffFixe * exp(expo);
				sommeFiltre += gaussienne[i][j][k];
			}
		}
	}

	
	/* copie de l'image source */
	ancienneImage = new unsigned char *[nbr];
	for (int k=0; k!=nbr; k++)
	{
		ancienneImage[k] = new unsigned char[largeur*hauteur*nbrBytes];
		memcpy(ancienneImage[k],donnees[k],largeur*hauteur*nbrBytes);
	}
	
	
	int hMin = (hauteurCoupe==-1)?0:hauteurCoupe;
	int hMax = (hauteurCoupe==-1)?nbr:hauteurCoupe+1;
	
	double *resultat;
	resultat = new double[largeur*hauteur];
	
	for (int k=hMin; k!=hMax; k++)
	{
		if (pos)
			(*pos)++;
		
		double valMin, valMax;
		double alpha, beta;
		valMin =  999999999;
		valMax = -999999999;		
		
		
		for (int j=0; j!=hauteur; j++)
		{
			for (int i=0; i!=largeur; i++)
			{
				resultat[j*largeur+i] = 0.0;
				
				for (int l=-hauteurFiltre; l<=hauteurFiltre; l++)
					for (int n=-largeurFiltre; n<=largeurFiltre; n++)
						for (int m=-profondeurFiltre; m<=profondeurFiltre; m++)
							resultat[j*largeur+i] +=  getValeurPixel(ancienneImage,k+m,i+n,j+l) * gaussienne[n+largeurFiltre][l+hauteurFiltre][m+profondeurFiltre];				
				
				resultat[j*largeur+i] /= sommeFiltre;				
				
				if (resultat[j*largeur+i] > valMax)
					valMax = resultat[j*largeur+i];
				else if (resultat[j*largeur+i] < valMin)
					valMin = resultat[j*largeur+i];
			}
		}
		
		
		/* gestion d'erreur : impossible de diviser par zro */
		if (valMin == valMax)
		{
			alpha = (255.0) / (double)(valMax);
			beta  = - valMin * alpha;
		}
		else
		{
			alpha = (255.0) / (double)(valMax-valMin);
			beta  = - valMin * alpha;
		}
		
		/* expansion dynamique */
		for (int j=0; j<hauteur; j++)
		{
			for (int i=0; i<largeur; i++)
			{				
				/*donnees[k][(j*largeur+i)*nbrBytes+(nbrBytes-1)] = (char)(alpha*resultat[j*largeur+i] + beta);			*/
				setValeurPixel(k,i,j,(int)(alpha*resultat[j*largeur+i] + beta));
			}
		}
		
		chercherBorne(k);
	}
	delete [] resultat;
	
	
	/* libration de la mmoire utilise pour sauvegarder l'image source */
	for (int i=0; i!=nbr; i++)
		delete [] ancienneImage[i];
	
	delete [] ancienneImage;
	
	
	/* libration de la mmoire utilise par le filtre gaussien */
	for (int i=0; i!=tailleX; i++)
	{
		for (int j=0; j!=tailleY; j++)
			delete [] gaussienne[i][j];
		
		delete [] gaussienne[i];
	}
	delete [] gaussienne;
}


void volume::morphoMaths (int hauteurCoupe, int ID, int *pos)
{
	unsigned char **save;
	int val, tmp;
	
	/* on rcupre le pointeur sur l'lment structurant */
	elementStruct *elementStructurant = elementStruct::getInstance();
	/* taille de la matrice */
	int taille = elementStructurant->getTailleElement();
	taille = (taille-1)/2;
	
	/* allocation/recopie du volume */
	save = new unsigned char *[nbr];
	for (int img=0; img!=nbr; img++)
	{
		save[img] = new unsigned char[largeur*hauteur*nbrBytes];
		memcpy(save[img],donnees[img],largeur*hauteur*nbrBytes);
	}
	
	/* 2d ou 3d ? */
	bool en3d = (hauteurCoupe==-1);
	int  hMin = en3d ? 0   : hauteurCoupe;
	int  hMax = en3d ? nbr : hauteurCoupe+1;
	
	for (int img=hMin; img!=hMax; img++)  /* image   */
	{
		for (int i=0; i<largeur; i++)     /* largeur */
		{
			for (int j=0; j<hauteur; j++) /* hauteur */
			{
				val = getValeurPixel(save,img,i,j);
				
				for (int k=-taille; k<=taille; k++)         /* dcalage image   */
				{
					for (int l=-taille; l<=taille; l++)     /* dcalage largeur */
					{
						for (int m=-taille; m<=taille; m++) /* dcalage hauteur */
						{
							if (elementStructurant->getValeur(l,m,k,en3d))
							{
								tmp = getValeurPixel(save,img+k,i+l,j+m);
								
								if (ID==ID_EROSION && tmp<val)
									val = tmp;
								else if (ID==ID_DILATATION && tmp>val)
									val = tmp;
							}
						}
					}
					
					if (!en3d)
						k = taille;
				}
				
				setValeurPixel(img, i, j, val);
			}
		}
		
		if (pos)
			(*pos)++;
	}
	
	
	/* libration de la mmoire utilise par l'image sauvegarde */	
	delete [] save;
}


void volume::dilatation (int hauteurCoupe, int *pos)
{
	morphoMaths(hauteurCoupe, ID_DILATATION, pos);
}


void volume::erosion (int hauteurCoupe, int *pos)
{
	morphoMaths(hauteurCoupe, ID_EROSION, pos);
}


/*---------------------[ TOUTE IMAGE BINAIRE                    ]---------------
 * Affecte une couleur  tous les pixels de la composante courante
 *----------------------------------------------------------------------------*/
void volume::peindre (int i, int j, int k, unsigned int couleurDepart, unsigned int couleurFin)
{
    if (i<0 || i>=largeur)
		return;
	
    if (j<0 || j>=hauteur)
		return;
	
    if (k<0 || k>=nbr)
		return;
	
	if (getValeurPixel(k,i,j) == couleurDepart)
	{ 
		setValeurPixel(k,i,j,couleurFin);
		
		peindre(i+0,j+0,k-1,couleurDepart,couleurFin);
		peindre(i+1,j+0,k+0,couleurDepart,couleurFin);
		peindre(i-1,j+0,k+0,couleurDepart,couleurFin);
		peindre(i+1,j+1,k+0,couleurDepart,couleurFin);
		peindre(i+1,j-1,k+0,couleurDepart,couleurFin);
	}
}


void volume::suppressionParasite (int *pos)
{
	if (MODE_VERBOSE)
		cout << "[+] Suppression des parasites." << endl;
	
	for (int i=0; i!=largeur; i++)
	{
		for (int j=0; j!=hauteur; j++)
		{
			peindre(i,j,0,BLANC,GRIS);
			peindre(i,j,nbr-1,BLANC,GRIS);
		}
		
		for (int k=0; k!=nbr; k++)
		{
			peindre(i,0,k,BLANC,GRIS);
			peindre(i,hauteur-1,k,BLANC,GRIS);
		}
	}
	
	
	for (int j=0; j!=hauteur; j++)
	{		
		for (int k=0; k!=nbr; k++)
		{
			peindre(0,j,k,BLANC,GRIS);
			peindre(largeur-1,j,k,BLANC,GRIS);
		}
	}
	
	
	/* on supprime tout ce qui n'a pas t colori */
	for (int k=0; k!=nbr; k++)
	{
		for (int i=0; i<largeur; i++)
			for (int j=0; j!=hauteur; j++)
				if (getValeurPixel(k,i,j) == BLANC)
					setValeurPixel(k,i,j,NOIR);
		
		if (k%2 && pos)
			(*pos)++;
	}
	
	
	/* le gris redevient blanc */
	for (int k=0; k!=nbr; k++)
	{
		for (int i=0; i<largeur; i++)
			for (int j=0; j!=hauteur; j++)
				if (getValeurPixel(k,i,j) == GRIS)
					setValeurPixel(k,i,j,BLANC);
		
		if (k%2 && pos)
			(*pos)++;
	}
}


void volume::boucherTrous ()
{
	int i,j,k;
	i = j = k = 0;
	while (getValeurPixel(k,i,j) != NOIR)
	{
		i++;
		if (i == largeur)
		{
			i = 0;
			j++;
			
			if (j == hauteur)
			{
				j = 0;
				k++;
			}
		}
	}
	peindre(i,j,k,NOIR,GRIS);
	
	
	/* le gris redevient blanc */
	for (i=0; i!=largeur; i++)
	{
		for (j=0; j!=hauteur; j++)
		{
			for (k=0; k!=nbr; k++)
			{
				if (getValeurPixel(k,i,j) == GRIS)
					setValeurPixel(k,i,j,NOIR);
				else
					setValeurPixel(k,i,j,BLANC);
			}
		}
	}	
}


void volume::contour (int kMin, int *pos)
{
	int kMax;
	if (kMin == -1)
	{
		kMin = 0;
		kMax = nbr;
	}
	else
		kMax = kMin+1;
		
	
	unsigned char **save;
	save = new unsigned char *[nbr];
	for (int img=0; img!=nbr; img++)
	{
		save[img] = new unsigned char[largeur*hauteur*nbrBytes];
		memcpy(save[img],donnees[img],largeur*hauteur*nbrBytes);
	}
	
	erosion();
	
	
	for (int k=kMin; k!=kMax; k++)
	{
		for (int i=0; i<largeur; i++)
			for (int j=0; j!=hauteur; j++)
				if (getValeurPixel(k,i,j) == BLANC)
					setValeurPixel(save,k,i,j,NOIR);
		
		if (pos!=NULL && k%2)
			(*pos)++;
	}	
	
	if (donnees)
	{
		for (int i=0; i<nbr; i++)
			if (donnees[i] != NULL)
				delete [] donnees[i];
		
		delete [] donnees;
	}
	donnees = save;
	return;
	
	/* le gris redevient blanc */
	for (int i=0; i<largeur; i++)
		for (int j=0; j!=hauteur; j++)
			for (int k=0; k!=nbr; k++)
				if (getValeurPixel(k,i,j) == BLANC)
					setValeurPixel(save,k,i,j,NOIR);
	
	
	if (donnees)
	{
		for (int i=0; i<nbr; i++)
			if (donnees[i] != NULL)
				delete [] donnees[i];
		
		delete [] donnees;
	}
	donnees = save;
}


bool volume::saveContour (int img, int i, int j, int *cpt, int *combien, string *listePts)
{
	if (getValeurPixel(img,i,j) != BLANC)
		return false;
	
	char tmp[255];
	if ((*combien) == 2)
	{
		(*combien) = 0;
		(*cpt)++;
		sprintf(tmp, "%.2f %.2f\n", (float)i, (float)j);
		(*listePts) = tmp + (*listePts);
	}
	else
		(*combien)++;
	
	setValeurPixel(img,i,j,NOIR);
	
	if (!saveContour(img,i+0,j+1,cpt,combien,listePts))
		if (!saveContour(img,i+1,j+1,cpt,combien,listePts))
			if (saveContour(img,i+1,j+0,cpt,combien,listePts))
				if (saveContour(img,i+1,j-1,cpt,combien,listePts))
					if (saveContour(img,i+0,j-1,cpt,combien,listePts))
						if (saveContour(img,i-1,j-1,cpt,combien,listePts))
							if (saveContour(img,i-1,j+0,cpt,combien,listePts))
								saveContour(img,i-1,j+1,cpt,combien,listePts);
	
	return true;
}


void volume::sauvegarderPoints (string fichier, string masque)
{
	setlocale (LC_ALL, "C");
	/************************* save des pts **********************/
	ofstream savePts (fichier.c_str(), ios::out);
	if (!savePts.is_open())
		cout << "[-] volume::sauvegarderPoints > Impossible d'ouvrir le fichier " << fichier << endl;
	else
	{
		for (int i=0; i!=nbr; i++)
		{			
			for (int j=0; j<hauteur; j++)
			{				
				for (int k=0; k<largeur; k++)
				{
					if (getValeurPixel(i,k,j) == BLANC)
					{
						string tmp;
						unsigned int pos;
						ostringstream x, y, z;
						x << ((float)k-largeur/2.)/5.;
						y << ((float)j-hauteur/2.)/5.;
						z << (float)i;
						
						tmp = masque;
						while ((pos = tmp.find("{X}")) != string::npos)
							tmp.replace(pos,3,x.str());
						
						while ((pos = tmp.find("{Y}")) != string::npos)
							tmp.replace(pos,3,y.str());
						
						while ((pos = tmp.find("{Z}")) != string::npos)
							tmp.replace(pos,3,z.str());
						
						savePts << tmp << endl;
					}
				}
			}
		}
		
		savePts.close();
	}
}


unsigned int volume::getValeurPixel (int num, int l, int h)
{
	return getValeurPixel(donnees,num,l,h);
}


unsigned int volume::getValeurPixel (unsigned char **tab, int num, int l, int h)
{
	int val = 0;
	
	if (num < 0)
		num = -num;
	else if (num >= nbr)
		num -= nbr;
	
	if (l < 0)
		l = -l;
	else if (l >= largeur)
		l -= largeur;
	
	if (h < 0)
		h = -h;
	else if (h >= hauteur)
		h -= hauteur;
	
	for (int i=0; i!=nbrBytes; i++)
		val = 255*val + (int)getValeurOctet(tab,num,l,h,i);
	
	return (unsigned int)val;
}


unsigned int volume::getValeurOctet (int num, int l, int h, int octet)
{
	return getValeurOctet(donnees,num,l,h,octet);
}


unsigned int volume::getValeurOctet (unsigned char **tab, int num, int l, int h, int octet)
{
	return tab[num][(h*largeur+l)*nbrBytes+(nbrBytes-octet-1)];
}


void volume::setValeurPixel (int num, int l, int h, int val)
{
	setValeurPixel (donnees,num,l,h,val);
}


void volume::setValeurPixel (unsigned char **tab,int num, int l, int h, int val)
{
	if (num<0 || num>=nbr)
		return;
	
	if (l<0 || l>=largeur)
		return;
	
	if (h<0 || h>=hauteur)
		return;	
	
	
	for (int i=0; i!=nbrBytes; i++)
	{
		if (val <= 255)
		{
			if (val == -1)
				tab[num][(h*largeur+l)*nbrBytes+i] = (unsigned char)(255);
			else
			{
				tab[num][(h*largeur+l)*nbrBytes+i] = (unsigned char)(val);
				val = 0;
			}
		}
		else
		{
			tab[num][(h*largeur+l)*nbrBytes+i] = (unsigned char)(val%255);
			val /= 255;
		}
	}
}


unsigned char *volume::get (int hauteurCoupe)
{
	if (hauteurCoupe < 0)
		hauteurCoupe = 0;
	
	if (hauteurCoupe >= nbr)
		hauteurCoupe = nbr-1;
	
	expansionDynamique(hauteurCoupe, &imageTmp);
	return imageTmp;
}


unsigned char *	volume::getBrut (int hauteurCoupe)
{
	if (hauteurCoupe < 0)
		hauteurCoupe = 0;
	
	if (hauteurCoupe >= nbr)
		hauteurCoupe = nbr-1;
	
	return donnees[hauteurCoupe];
}
